/* Cabinet knob 

  Modeled after my original Blender version at:
     https://www.thingiverse.com/thing:1815022.
  This one accommodates a captured nut and bolt, and will hopefully 
  last longer than the friction-fit self-tapping-screw version.

  Inputs set the size of the available nut and bolt.
  You can also scale the overall knob size.
*/ 

include <tjw-scad/dfm.scad>;
use <tjw-scad/arrange.scad>;
use <tjw-scad/moves.scad>;
use <tjw-scad/primitives.scad>
use <tjw-scad/spline.scad>

// Measured constants

DOOR = 24;  // depth of the surface you're attaching the knob to

BOLT_LEN = 1.5*inch;  // from tip to base of head
BOLT_HOLE_D = 1*inch/6;  // actual diameter of the hole the bolt passes through
NUT_H = 0.1*inch;  // height of the nut along the axis of the bolt
NUT_D = 8.8;  // diameter of the circle that touches all outer vertices of the nut
NUT_FACES = 6;  // 6 for hex nut, 4 for square

PLAY = 3;  // mm to allow for the screw to be tightened, accomodating variation in door etc.

// Derived constants

KNOB_HEIGHT = 27.4;  // original model height
KNOB_SCALE = KNOB_HEIGHT / 27.4;

PART_LINE_Z = 22.75 * KNOB_SCALE;
D = 41 * KNOB_SCALE;

SHAFT_PEN = BOLT_LEN - DOOR;  // distance the bolt shaft penetrates the object
NUT_Z = SHAFT_PEN - NUT_H - PLAY;  // height of bottom of nut from Z=0.
NUT_HOLE_D = NUT_D + LOOSE_FIT;

// Basic profile, in XY with the hole in -Y.
path = [
  [16.3 / 2, 0 * KNOB_SCALE],
  [17.3 / 2, 6 * KNOB_SCALE],
  [22.0 / 2, 13.25 * KNOB_SCALE],
  [32.0 / 2, 18 * KNOB_SCALE],
  [40.0 / 2, 22 * KNOB_SCALE],
  [31.0 / 2, 26.6 * KNOB_SCALE],
  [1 / 2, 27.4 * KNOB_SCALE]
];

// Mockup to make it easy to edit the profile - view the X-Y plane.
module preview() {
  spline_pot(path, $fn=60, preview=1, inner_targets=path);
}

// Main shape, modeled centered with the hole facing down on X-Y.
module body() {
  spline_lathe(path, $fn=120);
}

// Hole for the bolt shaft, inserted from the far side of the door.
// Modeled centered on XY plane.
module shaft_hole() {
  cylinder(d=BOLT_HOLE_D, h=SHAFT_PEN + LOOSE_FIT, $fn=20);
}

// Hole to hole the nut tightly.
// The model is divided in two pieces, and the nut is inserted before they're glued together.
// Modeled centered on XY plane.
module nut_hole(h=NUT_H + LOOSE_FIT) {
  d = NUT_HOLE_D;
  regular_polygon_prism(NUT_FACES, outer_d=NUT_D + LOOSE_FIT, h=h);

  // taper down instead of ending abruptly - prints better.
  taper_h = d / 2;

  flipOver()
    cylinder(h=taper_h, $fn=NUT_FACES, d1=d, d2=0);
}

// Full object.

module knob() {
  difference() {
    body();
    nudgeDown()
      shaft_hole();

    // Make the nut hole go all the way up to the parting line.
    moveUp(NUT_Z)
      nut_hole(PART_LINE_Z - NUT_Z);
  }
}

// Slices and extra bits to accomodate printer limitations.

// Touches the door, and holds the nut against the bolt.
module knobStem() {
  moveUp(PART_LINE_Z)
    flipOver()
      trimUpper(D, PART_LINE_Z)
        knob();
}

// Glued onto the stem.  Separated to allow the nut to be inserted,
// and for smoother printing.
module knobCap() {
  moveDown(PART_LINE_Z)
    trimLower(D, -PART_LINE_Z)
      knob();
}

// Insert into the stem after the nut, to hold it still
// so there's something to screw the bolt against.
module nutShim() {
  difference() {
    regular_polygon_prism(NUT_FACES, 
      outer_d=NUT_HOLE_D - LOOSE_FIT,
      h=PART_LINE_Z - NUT_Z - NUT_H);
    nudgeDown()
      shaft_hole();
  }
}

// Presentation

arrangeLine(D) {
  // For debugging the spline path:
  // preview();

  // Whole object visualization with cutaway:
  trimFront(2*D)
    knob();

  // In printing positions:
  knobCap();
  knobStem();
  nutShim();
}