#REQUIRES -Version 2.0

<#  

	EZDOK Camera fox FSX restart script
    Copyright (C) 2013  Radek Henys

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.

.SYNOPSIS
	Script restarts EZDOK Camera Software (EZCA.exe) Add-On program for FSX when certain conditions are met. 
	These conditions are defined in the script.
.DESCRIPTION
	This script purpose is to restart EZDOK Camera Add-On Software for FSX when you are also using FreeTrack.
	In this case, EZDOK will crash after about 10-20 minutes. When I was almost done with this script and I was
	putting together a post for my page, I assured myself that there is memory leak in EZCA when Freetrack is
	providing head movement input by TrackIR interface. EZCA private memory bytes usage rises slowly to some threshold
	where we will get that memory access violation error. At my computer, this happens when it uses about 160MB of memory.
	
	Working solution for this issue was posted here: http://passionfsx.iblogger.org/?page_id=164
	
	I wrote this script to make that solution work without 3rd party software and because with my configuration
	(Windows 8, Nvidia graphics) there wese some issues with that solution. That issue is, that EZCA steals focus
	from FSX when it is starting. As it does, FSX gets minimized when in fullscreen mode, or is deactivated when 
	in windowed mode.
	
	While this script isn't perfect, it is the best I achived after about month of trying... :( The script was
	able (at least on my system) to sucessfully restore (maximize again) FSX when running in fullscreen mode. This
	takes couple of seconds due to graphics source rebuilding (but this is DirectX job). When running in windowed
	mode, the script is able to quickly return focus to FSX.
	
	The script have some options which drives what it actually does. See ScriptOptions enumeration and $SCRIPT_OPTIONS.
	
	What the script does is:
	1) Looks if FSX is running and quits if it does not.
	2) Depending on options, checks if FreeTrack is running, and possibly quits if it does not
	3) Check if it has valid path for EZCA executable
	4) Checks if EZCA is running. If EZCA was running or relevant option is set, script continues to decide
	   if EZCA should be killed and restarted or not. If [ScriptOptions]::EZCA_RESTART_BY_MEM_LIMIT is enabled,
	   script will check EZCA memory consumption and compute average memory consumption growth rate. Based on this,
	   it will decide if memory consumption will be over threshold ($EZCA_PRIVATE_BYTES_LIMIT - EZCA_PRIVATE_BYTES_SMARGIN).
	   If memory usage will be too high by next time the script will run, script will continue to kill and restart EZCA, otherwise,
	   it will wait for that next execution.
	   After EZCA is killed nad started again, a waiting loop begins to wait for EZCA to initialize. The duration of that loop
	   is set by EZCA_AFTER_START_WAIT_TIME. I did not find any more reliable indication of that EZCA is ready.
	   Inside the waiting loop, depending on options, ezca priority ([ScriptOptions]::EZCA_MODIFY_PRIORITY option must be set) 
	   and affinity ([ScriptOptions]::EZCA_MODIFY_AFFINITY option must be set) may be set to predefined values (priority to: $EZCA_PRIORITY,
	   affinity to: $EZCA_AFFINITY).
	   If [ScriptOptions]::FSX_KEEP_FOREGROUND option is set, the script will detect if focus was stolen from FSX
	   and it will return it. If FSX running in fullscreen, it should get maximized again and if it is running in windowed mode,
	   it should get to foreground. Also, if running in windowed mode, you can make that window fullscreen by setting
	   FSX_MAKE_WINDOW_FULLSCREEN option. But this will not make FSX truly fullscreen. It will still run in windowed mode,
	   but the window borders and title bar will be removed and window maximized - thus it will cover whole screen and appear
	   to be fullscreen).
	   
	   If someone finds a way how to prevent FSX from minimizing, I will be happy if that person lets me know.
.NOTES
	File Name: ezca_restart.ps1
	Author: Radek Henys (admin@mouseviator.com)
	Prerequisite: PowerShell V2
 	Version: 1.0
	License: GPL v3
.LINK
	Script posted at:
	http://mouseviator.com
.EXAMPLE
	Use this script with silent_power_shell.vbs to define scheduled task. If you are going to use:
	[ScriptOptions]::EZCA_RESTART_BY_MEM_LIMIT (which is enabled by default), run the task more often, like
	every 2-4 minutes. Also, define this time at $TASK_INTERVAL so the memory growth rate can be calculated.
	The script will be executed more often, but EZCA will get restarted less often. If you are not going to
	use that option, set task interval for every 10-15 minutes or how long is EZCA able to run without crash 
	on your system.
#>

# *************************************************************************************************************
# **************** Constants, variables, functions and other required definitions *****************************
# *************************************************************************************************************

# These are used for logging
New-Variable SCRIPT_PATH -Value (Split-Path -parent $MyInvocation.MyCommand.Definition)
New-Variable LOG_FILE -Value (Join-Path $SCRIPT_PATH "ezca_restart.txt")

# Process names for FSX, EZCA and FreeTrack to find if they are running 
New-Variable FSX_PROCESS_NAME -Value "fsx" -Option ReadOnly
New-Variable EZCA_PROCESS_NAME -Value "ezca" -Option ReadOnly
New-Variable FREETRACK_PROCESS_NAME -Value "freetrack" -Option ReadOnly

# Here we construct path to EZCA executable. This expects default location at Program Files folder.
if ((Get-WmiObject Win32_OperatingSystem | Select-Object OSArchitecture) -match "64*") {
	New-Variable EZCA_EXECUTABLE -Value (Join-Path ([Environment]::GetFolderPath("ProgramFiles") + " (x86)") (Join-Path "EZCA" "ezca.exe")) -Option ReadOnly
} else {
	New-Variable EZCA_EXECUTABLE -Value (Join-Path ([Environment]::GetFolderPath("ProgramFiles")) (Join-Path "EZCA" "ezca.exe")) -Option ReadOnly
}

# These variables define priority and affinity we want EZCA process to have. You can modify these to fit
# your needs. Affinity is computed based on number of logical cores in a way that ezca will run just on 
# half of them. The equation used is: 2^N-1 where N is number of logical processors.
[System.Diagnostics.ProcessPriorityClass]$EZCA_PRIORITY = [System.Diagnostics.ProcessPriorityClass]::Normal
$processor_info = Get-WmiObject win32_Processor
New-Variable EZCA_AFFINITY -Value ([Convert]::ToInt32([System.Math]::Pow(2,[System.Math]::Ceiling($processor_info.NumberOfLogicalProcessors / 2)) - 1)) -Option ReadOnly

# Max EZCA private memory bytes limit in Kb. If it rises above this value, EZCA is about to crash
New-Variable EZCA_PRIVATE_BYTES_LIMIT -Value 153600 -Option ReadOnly
# Safety margin for EZCA private memory bytes consumption
New-Variable EZCA_PRIVATE_BYTES_SMARGIN -Value 600 -Option ReadOnly
# For how long we will record EZCA memory consumption
New-Variable EZCA_PRIVATE_BYTES_MEASURE_TIME -Value 4 -Option ReadOnly
# What is the time interval between task execution in seconds
New-Variable TASK_INTERVAL -Value 120 -Option ReadOnly

# This sets how long will be the wait loop after the EZCA process is started. On slower
# systems this should be longer, but it is a question of experimenting.
New-Variable EZCA_AFTER_START_WAIT_TIME -Value 5000 -Option ReadOnly

# Below we construct signature to import some Win32 functions we will use, define enumerations etc.
$signature = @"
	using System;
  	using System.Runtime.InteropServices;
  	
	//specify script options here
	public enum LogOptions
	{
		LOG_TO_CONSOLE = 1,
		LOG_TO_FILE = 2
	}
	
	public enum ScriptOptions 
	{
		/// <summary>
		/// Whether the script should modify EZCA priority
		/// </summary>
		EZCA_MODIFY_PRIORITY = 1,	
		/// <summary>
		/// Whether the script should modify EZCA affinity
		/// </summary>
		EZCA_MODIFY_AFFINITY = 2,
		/// <summary>
		/// Whether the script should EZCA even if it was not running. 
		/// NOTE: Due to preceding checks, FSX must be running and EZCA executable 
		///	path must me be also valid
		/// </summary>
		EZCA_START_IF_NOT_RUNNING = 4,
		/// <summary>
		/// Whether the EZCA should be restarted only when FreeTrack is running. (ie. script will
		///	terminate if Freetrack is not running)
		/// </summary>
		EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING = 8,
		/// <summary>
		/// If the script should make FSX window fullxscreen. (ie. This is NOT true fullscreen.
		/// This will only maximize FSX window to cover whole screen if it is running in windowed
		/// mode)
		/// </summary>
		FSX_MAKE_WINDOW_FULLSCREEN = 16,
		/// <summary>
		/// Whether the script should restore FSX window if is no longer in foreground.
		/// </summary>
		FSX_KEEP_FOREGROUND = 32,
		/// <summary>
		/// Whether the script should decide whether to kill EZCA by its memory usage.
		/// See $EZCA_PRIVATE_BYTES_LIMIT, $EZCA_PRIVATE_BYTES_SMARGIN, $EZCA_PRIVATE_BYTES_MEASURE_TIME
	    /// and $TASK_INTERVAL and step 5.1
		/// </summary>
		EZCA_RESTART_BY_MEM_LIMIT = 64
	}
	
	public enum ShowWindowCommands : int
	{
		/// <summary>
		/// Hides the window and activates another window.
		/// </summary>
		Hide = 0,
		/// <summary>
		/// Activates and displays a window. If the window is minimized or 
		/// maximized, the system restores it to its original size and position.
		/// An application should specify this flag when displaying the window 
		/// for the first time.
		/// </summary>
		Normal = 1,
		/// <summary>
		/// Activates the window and displays it as a minimized window.
		/// </summary>
		ShowMinimized = 2,
		/// <summary>
		/// Maximizes the specified window.
		/// </summary>
		Maximize = 3, // is this the right value?
		/// <summary>
		/// Activates the window and displays it as a maximized window.
		/// </summary>       
		ShowMaximized = 3,
		/// <summary>
		/// Displays a window in its most recent size and position. This value 
		/// is similar to <see cref="Win32.ShowWindowCommand.Normal"/>, except 
		/// the window is not activated.
		/// </summary>
		ShowNoActivate = 4,
		/// <summary>
		/// Activates the window and displays it in its current size and position. 
		/// </summary>
		Show = 5,
		/// <summary>
		/// Minimizes the specified window and activates the next top-level 
		/// window in the Z order.
		/// </summary>
		Minimize = 6,
		/// <summary>
		/// Displays the window as a minimized window. This value is similar to
		/// <see cref="Win32.ShowWindowCommand.ShowMinimized"/>, except the 
		/// window is not activated.
		/// </summary>
		ShowMinNoActive = 7,
		/// <summary>
		/// Displays the window in its current size and position. This value is 
		/// similar to <see cref="Win32.ShowWindowCommand.Show"/>, except the 
		/// window is not activated.
		/// </summary>
		ShowNA = 8,
		/// <summary>
		/// Activates and displays the window. If the window is minimized or 
		/// maximized, the system restores it to its original size and position. 
		/// An application should specify this flag when restoring a minimized window.
		/// </summary>
		Restore = 9,
		/// <summary>
		/// Sets the show state based on the SW_* value specified in the 
		/// STARTUPINFO structure passed to the CreateProcess function by the 
		/// program that started the application.
		/// </summary>
		ShowDefault = 10,
		/// <summary>
		///  <b>Windows 2000/XP:</b> Minimizes a window, even if the thread 
		/// that owns the window is not responding. This flag should only be 
		/// used when minimizing windows from a different thread.
		/// </summary>
		ForceMinimize = 11
	}
	 
	public enum WindowLongFlags : int
	{
		GWL_EXSTYLE = -20,
		GWLP_HINSTANCE = -6,
		GWLP_HWNDPARENT = -8,
		GWL_ID = -12,
		GWL_STYLE = -16,
		GWL_USERDATA = -21,
		GWL_WNDPROC = -4,
		DWLP_USER = 0x8,
		DWLP_MSGRESULT = 0x0,
		DWLP_DLGPROC = 0x4
	}	
	
	public enum SystemMetrics : int
	{
		 SM_CXSCREEN = 0,
		 SM_CYSCREEN = 1
	}	
	
	public class Win32 {
		// Window Styles 
		const UInt32 WS_OVERLAPPED = 0;
		const UInt32 WS_POPUP = 0x80000000;
		const UInt32 WS_CHILD = 0x40000000;
		const UInt32 WS_MINIMIZE = 0x20000000;
		const UInt32 WS_VISIBLE = 0x10000000;
		const UInt32 WS_DISABLED = 0x8000000;
		const UInt32 WS_CLIPSIBLINGS = 0x4000000;
		const UInt32 WS_CLIPCHILDREN = 0x2000000;
		const UInt32 WS_MAXIMIZE = 0x1000000;
		const UInt32 WS_CAPTION = 0xC00000;      
		const UInt32 WS_BORDER = 0x800000;
		const UInt32 WS_DLGFRAME = 0x400000;
		const UInt32 WS_VSCROLL = 0x200000;
		const UInt32 WS_HSCROLL = 0x100000;
		const UInt32 WS_SYSMENU = 0x80000;
		const UInt32 WS_THICKFRAME = 0x40000;
		const UInt32 WS_GROUP = 0x20000;
		const UInt32 WS_TABSTOP = 0x10000;
		const UInt32 WS_MINIMIZEBOX = 0x20000;
		const UInt32 WS_MAXIMIZEBOX = 0x10000;
		const UInt32 WS_TILED = WS_OVERLAPPED;
		const UInt32 WS_ICONIC = WS_MINIMIZE;
		const UInt32 WS_SIZEBOX = WS_THICKFRAME;	
			
		
		[DllImport("user32.dll")]
		[return: MarshalAs(UnmanagedType.Bool)]
		public static extern bool SetForegroundWindow(IntPtr hWnd);
		
		[DllImport("user32.dll")]
	 	public static extern IntPtr GetForegroundWindow();
		
		[DllImport("user32.dll")]
	 	public static extern int SetWindowLong(IntPtr hWnd, int nIndex, int dwNewLong);
		
		[DllImport("user32.dll")]
	 	[return: MarshalAs(UnmanagedType.Bool)]
	 	public static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);
		
		[DllImport("user32.dll")]
	 	public static extern int GetSystemMetrics(int nIndex);
		
		[DllImport("user32.dll")]
		[return: MarshalAs(UnmanagedType.Bool)]
		public static extern bool GetWindowRect(IntPtr hWnd, out RECT lpRect);
	}

	public struct RECT
	{
	    public int Left;        // x position of upper-left corner
	    public int Top;         // y position of upper-left corner
	    public int Right;       // x position of lower-right corner
	    public int Bottom;      // y position of lower-right corner
	}
"@

# Import what we defined above
Add-Type $signature

# Here we set default script options. Modify these to fit your needs
# These work as binary mask, setting corresponding bit to 1 to enable option,
# setting correspondig bit to 0 to disable option
New-Variable SCRIPT_OPTIONS -Value ([ScriptOptions]::EZCA_START_IF_NOT_RUNNING -bor
									[ScriptOptions]::FSX_KEEP_FOREGROUND -bor
									[ScriptOptions]::EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING -bor
									[ScriptOptions]::EZCA_RESTART_BY_MEM_LIMIT)

# 2) **** Define Functions *****
<#
.SYNOPSIS
	This is simple function used to write log messages.
.DESCRIPTION
	This function is used to write log messages. These can be written to console (by setting -LogToConsole $true) and/or 
	file (by setting -LogToFile $true and $LogFile <path to save log file>). Logging to console and file is enabled by default.
	The file is saved to the directory where the script is with the same name as the scrip have, but with txt extension.
.EXAMPLE
	Write-Message "This is my message"
#>
function Write-Message($Message,$Log=([LogOptions]::LOG_TO_CONSOLE -bor [LogOptions]::LOG_TO_FILE),$LogFile=$LOG_FILE) {
	if ($Log -band [LogOptions]::LOG_TO_CONSOLE) {
		Write-Host $Message
	}
	if ($Log -band [LogOptions]::LOG_TO_FILE) {
		$Message | Out-File -FilePath $LogFile -Append
	}
}

<#
.SYNOPSIS
	This function checks if process main window is in fullscreen mode.
.DESCRIPTION
	This function checks if process main window is in fullscreen mode. It doe so by compairing screen resolution
	with window size. If they are the same, window is considered to be fullscreen.
.EXAMPLE
	Is-Fullscreen -Process (Get-Process notepad)
.EXAMPLE 
	$process = Get-Process notepad
	Is-Fullscreen -Process $process
#>
function Is-Fullscreen
{
	param (
		[Parameter(Position=0,Mandatory=$true)]$Process
	)
	
	#get screen resolution
	$screen_width = [Win32]::GetSystemMetrics([SystemMetrics]::SM_CXSCREEN)
	$screen_height = [Win32]::GetSystemMetrics([SystemMetrics]::SM_CYSCREEN)
	
	$hWnd = $Process.MainWindowHandle
	if ($hWnd -ne 0) {
		#get size of window
		$rect = New-Object RECT
		$result = [Win32]::GetWindowRect($hWnd, [ref]$rect)
		$win_width = $rect.Right - $rect.Left
		$win_height = $rect.Bottom - $rect.Top
		
		return (($screen_width -eq $win_width) -and ($screen_height -eq $win_height))
	}
	return $false
}

<#
.SYNOPSIS
	This function switches process window to fullscreen mode.
.DESCRIPTION
	This function switches process window to fullscreen mode. It does so by removing window borders
	and title bar and then maximizing it.
.EXAMPLE
	Make-FullscreenWindow -Process (Get-Process notepad)
.EXAMPLE 
	$process = Get-Process notepad
	Make-FullscreenWindow -Process $process
#>
function Make-FullscreenWindow
{
	param (
		[Parameter(Position=0,Mandatory=$true)]$Process
	)
	
	$hWnd = $Process.MainWindowHandle
	if ($hWnd -ne 0) {
		Write-Message "Making main window with handle: $hWnd of process: $($Process.Name) PID($($Process.Id)) Fullscreen."
		# remove window frame and caption
		[Win32]::SetWindowLong($hWnd, [WindowLongFlags]::GWL_STYLE, [Win32]::WS_POPUP)
		# maximize window
		[Win32]::ShowWindow($hWnd, [ShowWindowCommands]::Maximize)
	}
}

<#
.SYNOPSIS
	This function checks if process main window is in foreground.
.DESCRIPTION
	This function checks if process main window is in foreground.
.EXAMPLE
	Is-Foreground -Process (Get-Process notepad)
.EXAMPLE 
	$process = Get-Process notepad
	Is-Foreground -Process $process
#>
function Is-Foreground
{
	param (
		[Parameter(Position=0,Mandatory=$true)]$Process
	)
	
	$focus_hWnd = [Win32]::GetForegroundWindow()
	$hWnd = $Process.MainWindowHandle
	if (($focus_hWnd -ne 0)-and ($hWnd -ne 0)) {
		return ($focus_hWnd -eq $hWnd)
	}
	return $false
}

<#
.SYNOPSIS
	This function brings process main window to foreground.
.DESCRIPTION
	This function will bring main window of given process to foreground.
.EXAMPLE
	Set-Foreground -Process (Get-Process notepad)
.EXAMPLE 
	$process = Get-Process notepad
	Set-Foreground -Process $process
#>
function Set-Foreground
{
	param(
		[Parameter(Position=0,Mandatory=$true)]$Process
	)
	
	$hWnd = $process.MainWindowHandle
	if ($hWnd -ne 0) {
		Write-Message "Bringing process: $($Process.Name) PID($($Process.Id)) main window: $hWnd to foreground." 
		[Win32]::SetForegroundWindow($hWnd)
	} else {
		Write-Message "Cannot bring window to foreground! Window handle is 0."
	}
}

# *************************************************************************************************************
# *********************** Script BEGINS here! *****************************************************************
# *************************************************************************************************************

#1) Delete old log file
if (Test-Path $LOG_FILE) {
	Remove-Item -Path $LOG_FILE
}

Clear-Host
Write-Message "Script started at: $(Get-Date)"
#2) Test if FSX is running
Write-Message "Looking for process: $FSX_PROCESS_NAME"
$fsx_process = $null
$fsx_process = Get-Process -Name $FSX_PROCESS_NAME -ErrorAction SilentlyContinue
if (!$fsx_process) {
	Write-Message "Process: $FSX_PROCESS_NAME NOT found! No job for me than. Bye"
	Write-Message "Script ended at: $(Get-Date)"
	exit
}

#3) Test if EZCA should be restarted only when FreeTrack is running and therefore, if FreeTrack is actually
#	running. Because if it does not, there is no need for restarting EZCA
if ($SCRIPT_OPTIONS -band [ScriptOptions]::EZCA_RESTART_ONLY_IF_FREETRACK_RUNNING) {
	Write-Message "Looking for process: $FREETRACK_PROCESS_NAME" 
	$freetrack_process = Get-Process -Name $FREETRACK_PROCESS_NAME -ErrorAction SilentlyContinue
	if (!$freetrack_process) {
		Write-Message "Process: $FREETRACK_PROCESS_NAME NOT found! No job for me than. Bye"
		Write-Message "Script ended at: $(Get-Date)"
		exit
	}
}

#4) Test if EZCA executable is valid path
if ((Test-Path -Path $EZCA_EXECUTABLE) -ne $true) {
	Write-Message "EZCA executable was not found at: $EZCA_EXECUTABLE ! No job for me than. Bye"
	Write-Message "Script ended at: $(Get-Date)"
	exit
}

#5) Test if EZCA is running. If it is, we will kill it and start it again. If it is not, but 
#   [ScriptOptions]::EZCA_START_IF_NOT_RUNNING option is set, we will start EZCA too. (We got here
#   through earlier test, so FSX is for sure running.)
Write-Message "Looking for process: $EZCA_PROCESS_NAME"
$ezca_process = $null
$ezca_process = Get-Process -Name $EZCA_PROCESS_NAME -ErrorAction SilentlyContinue
if ($ezca_process -or ($SCRIPT_OPTIONS -band [ScriptOptions]::EZCA_START_IF_NOT_RUNNING)) {

	# 5.1) This condition will check if EZCA private bytes memory usage is rising. It does when FreeTrack is running
	# and providing data using TrackIR interface. When private bytes reaches certain treshold, EZCA will crash.
	# Here we will compute rate of growth and decode whether there is enough time to wait for next script execution.
	# If there is time or the memory usage NOT rising, we will terminate script and wait for next execution. If there is
	# no time, we will let the script continue.
	if ($ezca_process -and ($SCRIPT_OPTIONS -band [ScriptOptions]::EZCA_RESTART_BY_MEM_LIMIT)) {
		$pb1 = $ezca_process.PrivateMemorySize64
		Start-Sleep -Seconds $EZCA_PRIVATE_BYTES_MEASURE_TIME
		$pb2 = $ezca_process.PrivateMemorySize64
		if ($pb2 -gt $pb1) {
			# what is the rate of growth of memory usage per second
			$pb_growth_rate = ($pb2 - $pb1) / $EZCA_PRIVATE_BYTES_MEASURE_TIME
			# how much will it probably rise until next script execution (in Kb)?
			$pb_future = $pb_growth_rate * $TASK_INTERVAL / 1024
			# Now we will check if future private bytes are still within limits
			$ezca_pb_future = ($ezca_process.PrivateMemorySize64 + $pb_future) / 1024
			$ezca_max_pb = $EZCA_PRIVATE_BYTES_LIMIT - $EZCA_PRIVATE_BYTES_SMARGIN
			if ($ezca_pb_future -gt $ezca_max_pb) {
				Write-Message "EZCA private bytes would be around: $ezca_pb_future by next time the script runs. That is more than safety value of: $ezca_max_pb .Continuing to restart EZCA..."
			} else {
				Write-Message "EZCA private bytes would be around: $ezca_pb_future by next time the script runs. That is less than safety value of: $ezca_max_pb . Will wait for next execution. Bye"
				Write-Message "Script ended at: $(Get-Date)"
				exit
			}
		} else {
			Write-Message "EZCA private bytes memory NOT rising! No job for me than. Bye"
			Write-Message "Script ended at: $(Get-Date)"
			exit
		}
	}


	#5.2) Kill EZCA if it is alive
	if ($ezca_process) {
		Write-Message "Killing process: $($ezca_process.Name) (PID: $($ezca_process.Id)) ..."
		$ezca_process.Kill()
		$ezca_process.WaitForExit()
		Write-Message "Process: $($ezca_process.Name) terminated. Exit code: $($ezca_process.ExitCode)"
		$ezca_process = $null
	}
	
	#5.3) start EZCA again
	Write-Message "Launching: $ezca_executable"
	
	$ezca_process = New-Object System.Diagnostics.Process
	
	$start_info = New-Object System.Diagnostics.ProcessStartInfo
	$start_info.FileName = $EZCA_EXECUTABLE
	$start_info.UseShellExecute = $false;
	$start_info.WindowStyle = [System.Diagnostics.ProcessWindowStyle]::Hidden
	
	$ezca_process.StartInfo = $start_info
	$ezca_process.Start()
	Write-Message "Launched: $($ezca_process.Name), PID: $($ezca_process.Id)"
	
	#5.4) Here we will start a loop of length $EZCA_AFTER_START_WAIT_TIME milliseconds. In this loop, according to
	#	  SCRIPT_OTIONS, we will modify EZCA priority, affinity and will try to keep FSX focused. (EZCA forces itself
	#	  to run on High priority when starting and this may stela focus from FSX which normally runs on Normal priority)
	Write-Message "Starting wait loop of: $($EZCA_AFTER_START_WAIT_TIME / 1000) seconds for: $($ezca_process.Name), PID: $($ezca_process.Id) to do its stuff."
	$loop_start_time = Get-Date
	Do {
		# Refresh process info
		$ezca_process.Refresh() 
		#5.4.1) Check ezca priority and reset to what we want
		if ($SCRIPT_OPTIONS -band [ScriptOptions]::EZCA_MODIFY_PRIORITY) {
			$curr_priority = $ezca_process.PriorityClass
			if ($curr_priority -ne $EZCA_PRIORITY) {
				try {
					$ezca_process.PriorityClass = $EZCA_PRIORITY
					Write-Message "Priority for process: $($ezca_process.Name) (PID:$($ezca_process.Id))  was set to: $($ezca_process.PriorityClass)"
				} catch {
					Write-Message "Failed to set priority for process: $($ezca_process.Name) (PID:$($ezca_process.Id)) due to following error:"
					Write-Message $_.Exception.Message
				}
			}
		}
		
		#5.4.2) Check ezca affinity and reset to what we want
		if ($SCRIPT_OPTIONS -band [ScriptOptions]::EZCA_MODIFY_AFFINITY) {
			$curr_affinity = $ezca_process.ProcessorAffinity
			if ($curr_affinity -ne $EZCA_AFFINITY) {
				try {
					$ezca_process.ProcessorAffinity = $EZCA_AFFINITY
					Write-Message "Affinity for process: $($ezca_process.Name) (PID:$($ezca_process.Id)) was set to: $($ezca_process.ProcessorAffinity)"
				} catch {
					Write-Message "Failed to set affinity for process: $($ezca_process.Name) (PID:$($ezca_process.Id)) due to following error:"
					Write-Message $_.Exception.Message
				}
			}
		}		

		#5.4.3) Check if focus was stolen from FSX and return it if desirable. If FSX is running in windowed mode
		#		and options says to make it fullscreen, we will make it fullscreen window (NOTE: This is
		#		the same as D3D fullscreen mode. FSX will still run in window mode, but the size will be the same
		#		as the size of the screen and border and title bar of the window will be removed.)
		if ($SCRIPT_OPTIONS -band [ScriptOptions]::FSX_KEEP_FOREGROUND) {
			if (!(Is-Foreground -Process $fsx_process)) {
				if (!(Is-Fullscreen -Process $fsx_process) -and 
					 ($SCRIPT_OPTIONS -band [ScriptOptions]::FSX_MAKE_WINDOW_FULLSCREEN)) {
					Make-FullscreenWindow -Process $fsx_process
				} else {
					Write-Message "Restoring window with handle: $($fsx_process.MainWindowHandle) for process: $($fsx_process.Name) PID($($fsx_process.Id))"
					[Win32]::ShowWindow($fsx_process.MainWindowHandle, [ShowWindowCommands]::Restore)
				}
				Set-Foreground -Process $fsx_process
			}
		}
		
		#wait a while
		Start-Sleep -Milliseconds 500
	} While (($(Get-Date) - $loop_start_time).TotalMilliseconds -le $EZCA_AFTER_START_WAIT_TIME)
	
	Write-Message "Wait loop finished!"
	Write-Message "All work done. Bye"
}

Write-Message "Script ended at: $(Get-Date)"