Include.addImports([
    "/JS/Services/Profile.js",
    "/Pages/Profile/Logbook/CSVHelper.js",
    "/templates/LogBookTable/EnhancedLogBookTable.html",
    "/templates/NewListButton/NewListButton.html",
    "/templates/searchElement/searchElement.html",    
], () => {
    window.customElements.define("logbook-page", LogbookPage);
});

class ExportCounters {
    constructor() {
        this.reset();
    }
    reset() {
        this.m_iNumMessagesSent = 0;
        this.m_iNumResponseOk = 0;
        this.m_iNumResponseNoData = 0;
        this.m_iNumResponseError = 0;
        this.m_iNumRequestsToSend = 0;
        this.m_iNumRequestsFailed = 0;
        this.m_iNumResponseReceived = 0;
    }
}

class ExportProcess {
    constructor() {
            this.reset();
    }
    reset() {
        this.m_bStarted = false;
        this.m_bFinished = false;
    }
}        


class LogbookPage extends TemplateElement {
    constructor() {
        super();
        this.updateLogbook = (_data) => {
            let table = document.querySelector(".logbookDivTable");
            table.setData(_data.logbook.entries, false, LogbookFilter.LOGBOOK_FILTERS_ALL);
            table.setTotal(_data.logbook.total);
            //set reference to the profile data for use in the export functions
            this.m_profileData = _data;            
            this.m_search.setResultNumber(this.visibleElementCount);
        };
        this.OnFilterChanged = (e) => {
            let table = document.querySelector(".logbookDivTable");
            var filtersElem = window.document.getElementById('filters');
            table.setFilter(filtersElem.getCurrentValue());
        };
        this.onCreateNote = (e) => {
            this.m_logbooklistener.createNote(e.detail.entryID, e.detail.content);
        };
        this.onFavoriteEntry = (e) => {
            this.m_logbooklistener.tooggleFavoriteEntry(e.detail.entryID);
        };
        this.onReflyEntry = (e) => {
            this.reflyLogEntryID = e.detail.entryID;
            let popUpParams = new PopUp.NotiticationParams();
            popUpParams.title = "TT:MENU.POPUP.LOGBOOK_REFLY_TITLE";
            popUpParams.contentData = "TT:MENU.POPUP.LOGBOOK_REFLY_DESC";
            popUpParams.description = "TT:MENU.POPUP.LOGBOOK_REFLY_DESC";
            popUpParams.buttons.push(new NotificationButton("TT:MENU.YES", "WP_VALIDATE_REFLY_LOGBOOK"));
            popUpParams.buttons.push(new NotificationButton("TT:MENU.NO", "WP_CANCEL_REFLY_LOGBOOK"));
            popUpParams.style = "small";
            PopUp.showPopUp(popUpParams);
        };
        this.onConfirmReflyEntry = () => {
            this.m_logbooklistener.reflyEntry(this.reflyLogEntryID);
        };
        this.onSeachInput = () => {
            let matchingNodes = document.querySelectorAll('#Items .searchable[title*="' + this.m_search.searchString.trim().toLowerCase() + '"]');
            let unMatchingNodes = document.querySelectorAll('#Items .searchable:not([title*="' + this.m_search.searchString.trim().toLowerCase() + '"])');
            if (!this.m_search.searchString || this.m_search.searchString.trim() == "") {
                matchingNodes = document.querySelectorAll('#Items .searchable');
            }
            unMatchingNodes.forEach(el => {
                el.classList.add('hide');
                if (el.classList.contains('active')) {
                    this.logbookTable.hideNote();
                }
            });
            matchingNodes.forEach(el => el.classList.remove('hide'));
            this.m_search.setResultNumber(matchingNodes.length);
            this.logbookTable.m_listScroll.sendSizeUpdate();
        };
        /**
         * This function will be called when Export via Clipboard button is clicked, it will give an option to select format
         * @returns {undefined}
         */
        this.onExportLogbookViaClipboard = () => {        
            let popUpParams = new PopUp.NotiticationParams();
            popUpParams.title = "Select export format";        
            popUpParams.description = "Please select in which format to export the logbook data:";
            popUpParams.buttons.push(new NotificationButton("JSON", "WP_EXPORT_LOGBOOK_VIA_CLIPBOARD_AS_JSON"));
            popUpParams.buttons.push(new NotificationButton("CSV", "WP_EXPORT_LOGBOOK_VIA_CLIPBOARD_AS_CSV"));
            popUpParams.style = "small";
            PopUp.showPopUp(popUpParams);
        };   
        this.exportLogbookViaClipboardAsJSON = () => {
            this.exportLogbookViaClipboard(LogbookPage.EXPORT_FORMAT_JSON);
        };
        this.exportLogbookViaClipboardAsCSV = () => {
            this.exportLogbookViaClipboard(LogbookPage.EXPORT_FORMAT_CSV);
        };
        this.onExportViaNetwork = () => {
            this.exportViaWebsocket();
        }
    }
    get templateID() {
        return "LogbookScreenTemplate";
    }
    get visibleElementCount() {
        return this.querySelectorAll('#Items .searchable:not(.hide)').length;
    }
    connectedCallback() {
        super.connectedCallback();
        this.m_logbooklistener = RegisterProfileListener();
        this.m_logbooklistener.onProfileUpdated(this.updateLogbook);
        var filtersElem = window.document.getElementById('filters');
        this.logbookTable = document.querySelector(".logbookDivTable");
        this.m_search = this.querySelector('#searchFlight');
        TemplateElement.call(this.m_search, () => {
            this.m_search.setResultNumber(this.visibleElementCount);
            this.m_search.addEventListener('onInputChange', this.onSeachInput);
        });
        if (filtersElem) {
            filtersElem.addEventListener("OnValidate", this.OnFilterChanged);
            filtersElem.classList.remove('hide');
            TemplateElement.call(filtersElem, () => {
                let filtersData = new NewListButtonData();
                filtersData.daChoices.push(Coherent.translate("TT:MENU.LOGBOOK_FILTERS_ALL"));
                filtersData.daChoices.push(Coherent.translate("TT:MENU.LOGBOOK_FILTERS_PARTIAL_FLIGHTS"));
                filtersData.daChoices.push(Coherent.translate("TT:MENU.LOGBOOK_FILTERS_GROUND_TO_GROUND"));
                filtersData.daChoices.push(Coherent.translate("TT:MENU.LOGBOOK_FLIGHTS_ENDED"));
                filtersData.daChoices.push(Coherent.translate("TT:MENU.LOGBOOK_FLIGHTS_STARTED"));
                filtersElem.SetData(filtersData);
            });
        }
        this.logbookTable.addEventListener('createNote', this.onCreateNote);
        this.logbookTable.addEventListener('favoriteEntry', this.onFavoriteEntry);
        this.logbookTable.addEventListener('reflyEntry', this.onReflyEntry);
        //add handler for copy to clipboard		
        this.m_cmdExportViaClipboard = document.querySelector('#exportLogbookViaClipboard');
        if (this.m_cmdExportViaClipboard) {
            this.m_cmdExportViaClipboard.addEventListener('OnValidate', this.onExportLogbookViaClipboard);
        }
        this.m_cmdExportViaNetwork = document.querySelector('#exportLogbookViaNetwork');
        if (this.m_cmdExportViaNetwork) {
            this.m_cmdExportViaNetwork.addEventListener('OnValidate', this.onExportViaNetwork);
        }
        this.m_exportOperationLog = document.querySelector('#logbookExportOperationLog');
        Coherent.on('WP_VALIDATE_REFLY_LOGBOOK', this.onConfirmReflyEntry);
        Coherent.on('WP_EXPORT_LOGBOOK_VIA_CLIPBOARD_AS_JSON', this.exportLogbookViaClipboardAsJSON);
        Coherent.on('WP_EXPORT_LOGBOOK_VIA_CLIPBOARD_AS_CSV', this.exportLogbookViaClipboardAsCSV);
    }         
    /**
     * A function to copy logbook data to clipboard. It will first try if browser (in this case - CoherentGT) supports navigator.clipboard object
     * and if so, will copy logbook as JSOn via this object. But current version does not support this!!! 
     * Second approach is to create hidden text field, paste the logbook JSON in it and then apply a copy command on it. This, so far, worked.
     * 
     * @param {string} format Format in which to export. Either "json" or "csv"
     * 
     * @returns {undefined}
     */
    exportLogbookViaClipboard(format = LogbookPage.EXPORT_FORMAT_JSON) {
        if (this.m_profileData.logbook) {
            let copied = false;
            let errorMsg = "";  //error message if we fail                                    

            //disable the copy to cliboard button
            this.enable(false);
            //The navigator.clipboard is not supported by Coherent, but keep it there, maybe it will be some time in the future
            if (navigator.clipboard) {
                try {
                    if (format === LogbookPage.EXPORT_FORMAT_JSON) {
                        navigator.clipboard.writeText(JSON.stringify(this.m_profileData.logbook));
                    } else if (format === LogbookPage.EXPORT_FORMAT_CSV) {
                        navigator.clipboard.writeText(CSVHelper.logbookToCSV(this.m_profileData.logbook));
                    }
                    copied = true;
                } catch (err) {
                    errorMsg = err.message;
                }
            } else {
                //This is fallback approach, using almost invisible text field and copy command
                let textArea = document.createElement("textarea");

                //
                // *** This styling is an extra step which is likely not required. ***
                //
                // Why is it here? To ensure:
                // 1. the element is able to have focus and selection.
                // 2. if the element was to flash render it has minimal visual impact.
                // 3. less flakyness with selection and copying which **might** occur if
                //    the textarea element is not visible.
                //
                // The likelihood is the element won't even render, not even a
                // flash, so some of these are just precautions. However in
                // Internet Explorer the element is visible whilst the popup
                // box asking the user for permission for the web page to
                // copy to the clipboard.
                //

                // Place in the top-left corner of screen regardless of scroll position.
                textArea.style.position = 'fixed';
                textArea.style.top = 0;
                textArea.style.left = 0;

                // Ensure it has a small width and height. Setting to 1px / 1em
                // doesn't work as this gives a negative w/h on some browsers.
                textArea.style.width = '2em';
                textArea.style.height = '2em';

                // We don't need padding, reducing the size if it does flash render.
                textArea.style.padding = 0;

                // Clean up any borders.
                textArea.style.border = 'none';
                textArea.style.outline = 'none';
                textArea.style.boxShadow = 'none';

                // Avoid flash of the white box if rendered for any reason.
                textArea.style.background = 'transparent';

                if (format === LogbookPage.EXPORT_FORMAT_JSON) {
                    textArea.value = JSON.stringify(this.m_profileData.logbook);
                } else if (format === LogbookPage.EXPORT_FORMAT_CSV) {                    
                    textArea.value = CSVHelper.logbookToCSV(this.m_profileData.logbook);
                }

                document.body.appendChild(textArea);
                textArea.focus();
                textArea.select();

                try {
                    copied = document.execCommand('copy');
                } catch (err) {
                    errorMsg = err.message;
                }

                //remove the temp textArea
                document.body.removeChild(textArea);
            }

            //did we succeed?
            if (copied) {
                this.logExportOperation("Logbook data has been copied to clipboard.");
                this.clearExportOperationLog();
            } else {
                //display error message
                this.logExportOperation("Failed to copy logbook data to clipboard! Reason: " + errorMsg, true, true);
                this.clearExportOperationLog(5000);

                /*let popUpParams = new PopUp.NotiticationParams();
                 popUpParams.title = Coherent.translate("TT:POPUP.LOGBOOK.COPY_TO_CLIPBOARD_FAILED_DT");
                 popUpParams.contentData = "";
                 popUpParams.description = Coherent.translate("TT:POPUP.LOGBOOK.COPY_TO_CLIPBOARD_FAILED_MSG") + " " + errorMsg;
                 //just ok button, no event needed, just hide the popup
                 popUpParams.buttons.push(new NotificationButton("TT:MENU.OK"));
                 popUpParams.style = "small";
                 PopUp.showPopUp(popUpParams);*/
            }

            //enable the copy to clipboard button
            this.enable(true);
        }
    }
    /**
     * A function to send logbook data over network using websocket. If websocket is not supported (or websock is not connected within 10 seconds), 
     * it fill try to use the {@link #exportViaNetworkXHR()} function.
     *  
     * @returns {undefined}
     */
    exportViaWebsocket() {
        let cmdExportViaNetworkButton = this;  //store reference to the button
        let logbookPage = this;

        if (this.m_profileData) {
            let bWebsocketSupported = ("WebSocket" in window);            

            //Mainly, we try using websockets
            if (bWebsocketSupported) {
                try {
                    logbookPage.logExportOperation("WebSocket is supported! Will try to use it...");                    
                    //prepare vraibales for export process monitoring
                    if (!logbookPage.m_ExportProcess) {
                        logbookPage.m_ExportProcess = new ExportProcess();
                    } else {
                        logbookPage.m_ExportProcess.reset();
                    }

                    //disable the send button
                    cmdExportViaNetworkButton.enable(false);

                    // Let us open a web socket
                    var ws = new WebSocket(LogbookPage.FSLE_WS_ADDRESS);
                    //init some data we need to know
                    ws.m_data = this.m_profileData;
                    ws.m_ExportCounters = new ExportCounters();

                    //On message callback
                    ws.addEventListener('message', function (event) {
                        //expect to receive JSON object as an answer
                        var response = JSON.parse(event.data);
                        if (response) {
                            console.log("Received: " + response);

                            switch (response.response_code) {
                                case 0:
                                    this.m_ExportCounters.m_iNumResponseOk++;
                                    break;
                                case 1:
                                    this.m_ExportCounters.m_iNumResponseNoData++;
                                    break;
                                default:
                                    this.m_ExportCounters.m_iNumResponseError++;
                            }
                        } else {
                            console.error("The websocket response was not expected JSON!")
                        }

                        this.m_ExportCounters.m_iNumResponseReceived++;

                        //close the socket after we recived response for all sent messages, or the connection will timeout...
                        if (this.m_ExportCounters.m_iNumResponseReceived >= this.m_ExportCounters.m_iNumRequestsToSend) {
                            ws.close(1000, "That's all folks!");
                        }
                    });

                    //Close callback
                    ws.addEventListener('close', function (event) {
                        //enable the send button
                        cmdExportViaNetworkButton.enable(true);
                        logbookPage.m_ExportProcess.m_bFinished = true;

                        // websocket is closed.
                        logbookPage.logExportOperation("Websocket connection closed. " + this.m_ExportCounters.m_iNumMessagesSent + " messages sent.", false);
                        logbookPage.logExportOperation("Logbook data export using Websocket finished!. " + this.m_ExportCounters.m_iNumMessagesSent + " data blocks sent, " + this.m_ExportCounters.m_iNumResponseOk + " data blocks received.", false);
                        logbookPage.clearExportOperationLog();
                        console.log("Logbook data export using Websocket finished! \nTotal number of data blocks sent is: " + this.m_ExportCounters.m_iNumMessagesSent +
                                "\nFailed data blocks: " + this.m_ExportCounters.m_iNumRequestsFailed +
                                "\nOK responses received: " + this.m_ExportCounters.m_iNumResponseOk +
                                "\nNO DATA responses received: " + this.m_ExportCounters.m_iNumResponseNoData +
                                "\nERROR responses received: " + this.m_ExportCounters.m_iNumResponseError);
                    });

                    //Error callback
                    ws.addEventListener('error', function (error) {
                        logbookPage.logExportOperation("The logbook export using Websocket FAILED. Reason: " + error.message, true, true);
                        this.m_iNumRequestsFailed = (this.m_ExportCounters.m_iNumRequestsToSend - this.m_ExportCounters.m_iNumMessagesSent);
                        //make sure connection is closed
                        ws.close(1001, "Ooops, something failed! On error callback was called! Error message: " + error.message);
                    });

                    //Open callback
                    ws.addEventListener('open', function (event) {
                        logbookPage.m_ExportProcess.m_bStarted = true;
                        // Web Socket is connected, send data using send()            
                        // send data by timeline entries                    
                        if (this.m_data.logbook.entries && this.m_data.logbook.entries.length) {
                            let websocket = this;
                            this.m_ExportCounters.m_iNumRequestsToSend = this.m_data.logbook.entries.length;

                            //send all data blocks
                            this.m_data.logbook.entries.forEach(tlEntry => {
                                let sTlEntry = JSON.stringify(tlEntry);
                                logbookPage.logExportOperation("Sending data block: " + (websocket.m_ExportCounters.m_iNumMessagesSent + 1) + " / " + websocket.m_ExportCounters.m_iNumRequestsToSend);
                                ws.send(sTlEntry);
                                //count how many messages we sent
                                websocket.m_ExportCounters.m_iNumMessagesSent++;
                            });
                        }
                    });

                    setTimeout(function () {
                        if (!logbookPage.m_ExportProcess.m_bStarted && !logbookPage.m_ExportProcess.m_bFinished && !cmdExportViaNetworkButton.enabled) {
                            //the button is still disabled, and the 
                            logbookPage.logExportOperation("Looks like Websocket connection FAILED! Will try XHR...");
                            logbookPage.exportViaNetworkXHR(ws.m_data);
                        }
                    }, 10000);
                } catch (err) {
                    console.error("Unexpected error while trying to conenct using websocket! Error: " + err.message);
                }
            }

            //if websocket not supported
            if (!bWebsocketSupported) {
                logbookPage.exportViaNetworkXHR(this.m_data);
            }
        }
    }
    /**
     * A function to export logbook data using XHR requests.
     * 
     * @param {type} _data Profile data containg the logbook data.
     * @returns {undefined}
     */
    exportViaNetworkXHR(_data) {
        let cmdExportViaNetworkButton = this.m_cmdExportViaNetwork;  //store reference to the button
        let logbookPage = this;
        //Try send data using XHR
        //disable the export button first
        cmdExportViaNetworkButton.enable(false);

        try {
            //Prepare export counters
            if (!logbookPage.m_ExportCounters) {
                logbookPage.m_ExportCounters = new ExportCounters();
            } else {
                logbookPage.m_ExportCounters.reset();
            }
            
            //prepare vraibales for export process monitoring
            if (!logbookPage.m_ExportProcess) {
                logbookPage.m_ExportProcess = new ExportProcess();
            } else {
                logbookPage.m_ExportProcess.reset();
            }

            if (_data.logbook.entries && _data.logbook.entries.length) {
                //New approach here, we send XHR request one by one. Next request is sent after we receive response
                //for the previous one. It is not good idea to start bambillion of requests in the loop.
                //It might not present problem if they go Good, but if theer is no connection for example, it would easily lock up the CoherentGT running the MSFS GUI
                //thus, the XHR requests will stop after second failed request
                var firstXHR = logbookPage.createExportXHRRequest(_data, logbookPage, 0);
                firstXHR.export();
            } else {
                logbookPage.logExportOperation("No Logbook data found!!!");
                logbookPage.clearExportOperationLog();
            }
        } catch (err) {
            logbookPage.logExportOperation("Logbook export using XHR FAILED! Reason: " + err.message, true, true);
            logbookPage.clearExportOperationLog(5000);
        }
    }
    /**
     * A helper function that build XHR request to send logbook time line entry using XHR. It accepte reference to the whole profile data, which includes
     * whole logbook and index of timle line entry to send. If than continues generating and sending other XHR requests until it sends all remaining
     * time line entries, or fails 2 times in row.
     * 
     * @param {type} _data Reference to profile data, where the logbook is
     * @param {type} logbookPage Reference to the logbookPage object, where other needed data are
     * @param {type} itemToSend Index of logbook time line entry to send
     * @returns {LogbookPage.createExportXHRRequest.xhr|XMLHttpRequest}
     */
    createExportXHRRequest(_data, logbookPage, itemToSend) {

        var xhr = new XMLHttpRequest();
        var url = LogbookPage.FSLE_XHR_ADDRESS;
        xhr.open("POST", url, true);
        xhr.setRequestHeader("Content-Type", "application/json;charset=utf-8");
        xhr.responseType = 'json'       //expecting JSON answer
        xhr.timeout = 8000;     //timeout after 2s
        xhr.m_ExportCounters = logbookPage.m_ExportCounters;    //store reference to export counters
        xhr.logbookPage = logbookPage;                          //reference to export button
        xhr.m_data = _data;                                     //store reference to data
        xhr.m_itemToSend = itemToSend;                          //store reference to current item to send            

        xhr.checkAllSendAndReceived = function () {
            //did we send all requests ?
            if ((this.m_ExportCounters.m_iNumMessagesSent + this.m_ExportCounters.m_iNumRequestsFailed) >= this.m_ExportCounters.m_iNumRequestsToSend) {
                //enable the send button
                this.logbookPage.m_cmdExportViaNetwork.enable(true);
                this.logbookPage.m_ExportProcess.m_bFinished = true;
                this.logbookPage.logExportOperation("Logbook data export using XHR finished!. " + this.m_ExportCounters.m_iNumMessagesSent + " data blocks sent, " + this.m_ExportCounters.m_iNumResponseOk + " data blocks received.", false);
                this.logbookPage.clearExportOperationLog(5000);
                console.log("Logbook data export using XHR finished! \nTotal number of data blocks sent is: " + this.m_ExportCounters.m_iNumMessagesSent +
                        "\nFailed data blocks: " + this.m_ExportCounters.m_iNumRequestsFailed +
                        "\nOK responses received: " + this.m_ExportCounters.m_iNumResponseOk +
                        "\nNO DATA responses received: " + this.m_ExportCounters.m_iNumResponseNoData +
                        "\nERROR responses received: " + this.m_ExportCounters.m_iNumResponseError);
            } else {
                //are there any other items to send?
                //are we failing?
                if (this.m_ExportCounters.m_iNumRequestsFailed >= 2) {
                    this.logbookPage.m_cmdExportViaNetwork.enable(true);
                    this.logbookPage.m_ExportProcess.m_bFinished = true;
                    this.logbookPage.logExportOperation("Export using XHR was aborted due to too many failures!", true, true);
                    this.logbookPage.clearExportOperationLog(5000);
                    console.log("Logbook data export using XHR finished (UNSUCCESSFULL)! \nTotal number of data blocks sent is: " + this.m_ExportCounters.m_iNumMessagesSent +
                            "\nFailed data blocks: " + this.m_ExportCounters.m_iNumRequestsFailed +
                            "\nOK responses received: " + this.m_ExportCounters.m_iNumResponseOk +
                            "\nNO DATA responses received: " + this.m_ExportCounters.m_iNumResponseNoData +
                            "\nERROR responses received: " + this.m_ExportCounters.m_iNumResponseError);    
                    
                    //display warning popup
                    let popUpParams = new PopUp.NotiticationParams();
                    popUpParams.title = Coherent.translate("Export FAILED!!!");
                    popUpParams.contentData = "";
                    popUpParams.description = Coherent.translate("Sorry, but it failed to export the logbook! At least 2 XHR requests FAILED!\n" +
                                                                 "It seems that there is no server to receive the data. \n" +
                                                                 "So, please make sure there is before trying again...\n" + 
                                                                 "During testing, accumulated unsuccessfull XHR requests caused\n" +
                                                                 "FREEZE/LOCK DOWN of the MSFS GUI (event though the requests are aborted!)");
                    //just ok button, no event needed, just hide the popup
                    popUpParams.buttons.push(new NotificationButton("TT:MENU.OK"));
                    popUpParams.style = "small";
                    PopUp.showPopUp(popUpParams);
                } else {
                    let nextItemToSend = this.m_itemToSend + 1;
                    if (nextItemToSend < this.m_ExportCounters.m_iNumRequestsToSend) {
                        var nextXHR = this.logbookPage.createExportXHRRequest(this.m_data, this.logbookPage, nextItemToSend);
                        nextXHR.export();
                    } else {
                        //fail switch, for the case we are not sending any more requests and the first if does not catch the end of export (which it always should)
                        //we statrt the function that will check in next 10 seconds if the variables determining end of pocess are set and button is enabled
                        var selfXhr = this;
                        setTimeout(function () {
                            if (!selfXhr.logbookPage.m_ExportProcess.m_bStarted && !selfXhr.logbookPage.m_ExportProcess.m_bFinished && !selfXhr.logbookPage.m_cmdExportViaNetwork.enabled) {
                                selfXhr.logbookPage.m_cmdExportViaNetwork.enable(true);
                            }
                        }, 1000);
                    }
                }
            }
        }

        xhr.export = function () {
            if (this.m_ExportCounters.m_iNumRequestsToSend == 0) {
                this.m_ExportCounters.m_iNumRequestsToSend = this.m_data.logbook.entries.length;
            }

            if (this.m_itemToSend < this.m_ExportCounters.m_iNumRequestsToSend) {
                let sTlEntry = JSON.stringify(this.m_data.logbook.entries[this.m_itemToSend]);
                //send the XHR request
                this.logbookPage.logExportOperation("Sending data block: " + (this.m_itemToSend + 1) + " / " + this.m_ExportCounters.m_iNumRequestsToSend);
                this.logbookPage.m_ExportProcess.m_bStarted = true;      //save that we started sending data - no matter now if it will succeed or not, need to know we are trying
                this.send(sTlEntry);
            }
        }

        xhr.addEventListener('readystatechange', event => {
            let selfXhr = event.target;
            //if received response headers, consider the XHR message as succesfully sent
            if (selfXhr.readyState == 2) {
                selfXhr.m_ExportCounters.m_iNumMessagesSent++;                
                //update progress
                selfXhr.logbookPage.logExportOperation("Sending data block: " + (selfXhr.m_ExportCounters.m_iNumMessagesSent) + " / " + selfXhr.m_ExportCounters.m_iNumRequestsToSend);
            }
        });
        xhr.addEventListener('load', event => {
            let selfXhr = event.target;
            if (selfXhr.status != 200) {
                console.error("The XHR response did return status other than 200. Returned status is: " + selfXhr.status);
                selfXhr.m_ExportCounters.m_iNumRequestsFailed++;
            } else {
                let responseObj = selfXhr.response;
                switch (responseObj.response_code) {
                    case 0:
                        selfXhr.m_ExportCounters.m_iNumResponseOk++;
                        break;
                    case 1:
                        selfXhr.m_ExportCounters.m_iNumResponseNoData++;
                        break;
                    default:
                        selfXhr.m_ExportCounters.m_iNumResponseError++;
                }
            }

            selfXhr.checkAllSendAndReceived();
        });

        xhr.addEventListener('error', error => {
            let selfXhr = error.target;
            //something went wrong
            console.error("Logbook export XHR request failed! Reason: " + error.message);
            selfXhr.m_ExportCounters.m_iNumRequestsFailed++;
            selfXhr.logbookPage.logExportOperation("Failed to send data block: " + (selfXhr.m_itemToSend + 1) + " / " + selfXhr.m_ExportCounters.m_iNumRequestsToSend);
            selfXhr.abort();
            selfXhr.checkAllSendAndReceived();
        });
        xhr.addEventListener('timeout', event => {
            //let selfXhr = event.target;
            //we timed out
            console.error("Logbook export XHR request timed out!");
            //selfXhr.m_ExportCounters.m_iNumRequestsFailed++;
            //error callback should be called next...
        });

        return xhr;
    }    
    /**
     * A helper function to log given message to export operation element - as text, showing it ot user and also to console/log
     * 
     * @param {String} message A message to log
     * @param {boolean} bLog Whether to log message to log/console too
     * @param {boolean} bError Whether this is error message or not.
     * @returns {undefined}
     */
    logExportOperation(message, bLog = true, bError = false) {
        if (this.m_exportOperationLog) {
            this.m_exportOperationLog.innerHTML = message;
            if (bError) {
                this.m_exportOperationLog.classList.add("error_message");
            } else {
                this.m_exportOperationLog.classList.remove("error_message");
            }
        }
        if (bLog) {
            if (Logger) {
                if (bError) {
                    Logger.error(message);
                } else {
                    Logger.info(message);
                }
            } else {
                if (bError) {
                    console.error(message);
                } else {
                    console.log(message);
                }
            }
    }
    }
    /**
     * This function will clear the export operation element text after given timeout.
     * 
     * @param timeout A timout to clear the element text.
     * @returns {undefined}
     */
    clearExportOperationLog(timeout = 2000) {
        let logbookPage = this;

        setTimeout(function () {
            if (logbookPage.m_exportOperationLog) {
                logbookPage.m_exportOperationLog.innerHTML = "";
                logbookPage.m_exportOperationLog.classList.remove("error_message");
            }
        }, timeout);
    }
    static get FSLE_WS_ADDRESS() {
        return "ws://127.0.0.1:9448/fsle_msfs_events/*";
    }
    static get FSLE_XHR_ADDRESS() {
        return "http://127.0.0.1:9448/fsle_msfs_events/*";
    }
    static get EXPORT_FORMAT_JSON() {
        return "json";
    }
    static get EXPORT_FORMAT_CSV() {
        return "csv";
    }
}
checkAutoload();
//# sourceMappingURL=Logbook.js.map